﻿using System.Collections.ObjectModel;
using System.Text;

namespace MCalcConsole
{
    internal class StackEntries : ObservableCollection<StackEntry>
    {
        public VirtualStack Values = new VirtualStack();
        public const int CMod = 26;

        // Returns absolute reference to entry by address from end of stack
        public SRSingle GetRefByCIndex(int ct, char lt)
        {
            int l = this.Count;
            int bp = l % CMod;
            int q = 'Z' - lt;
            if (q >= bp)
            {
                ct++;
            }
            int ir = l - bp - (ct * CMod) + q;

            if (ir < 0 || ir >= l)
                throw new ArgumentOutOfRangeException("Specified index does not exist in stack");
            return new SRSingle() { stackIndex = ir };
        }

        /// <summary>
        /// Returns absolute reference to entry relative to end of stack
        /// </summary>
        /// <param name="ai">1 for most recent entry, 2+ for older entries</param>
        /// <returns>Single line reference</returns>
        /// <exception cref="ArgumentOutOfRangeException">Index outside stack bounds</exception>
        public SRSingle GetRefByRelativeIndex(int ai)
        {
            if (ai < 1 || ai > this.Count)
                throw new ArgumentOutOfRangeException("Specified relative index does not exist in stack");
            return new SRSingle() { stackIndex = (this.Count - ai) };
        }

        public StackEntry GetByRef(SRSingle rf)
        {
            return this[rf.stackIndex];
        }

        public SRSingle GetLastCommand()
        {
            int i = this.Count - 1;
            while(i >= 0)
            {
                if (this[i] is CommandSE)
                    return new SRSingle() { stackIndex = i };
                i--;
            }
            throw new Exception("No command found in the stack");
        }

        public (SRSingle,SRSingle) GetLastIterator()
        {
            int i = this.Count - 1;
            int j = -1;
            while (i >= 0)
            {
                if (this[i] is CommandSE)
                {
                    if (((CommandSE)this[i]).mlin != 0)
                    {
                        if(j == -1)
                        {
                            j = i;
                        }
                    }
                    else
                    {
                        if (j != -1)
                        {
                            if (((CommandSE)this[i]).mlout == 0)
                            {
                                return (new SRSingle() { stackIndex = i }, new SRSingle() { stackIndex = j });
                            }
                        }
                    }
                }
                else
                {
                    if (j != -1)
                    {
                        throw new Exception("Incorrect iterator end in the stack (should never see this)");
                    }
                }
                i--;
            }
            throw new Exception("No iterator found in the stack");
        }

        public (SRSingle, SRSingle) GetLastMultiline()
        {
            int i = this.Count - 1;
            int j = -1;
            while (i >= 0)
            {
                if (this[i] is CommandSE)
                {
                    if (((CommandSE)this[i]).mlout != 0)
                    {
                        if (j == -1)
                        {
                            j = i;
                        }
                    }
                    else
                    {
                        if (j != -1)
                        {
                            return (new SRSingle() { stackIndex = i }, new SRSingle() { stackIndex = j });
                        }
                    }
                }
                else
                {
                    if (j != -1)
                    {
                        throw new Exception("Incorrect multiline end in the stack (should never see this)");
                    }
                }
                i--;
            }
            throw new Exception("No multiline found in the stack");
        }

        public SRSingle GetLastValue()
        {
            int i = this.Count - 1;
            while (i >= 0)
            {
                if (this[i] is ValueSE)
                    return new SRSingle() { stackIndex = i };
                i--;
            }
            throw new Exception("No value found in the stack");
        }

        public SRSingle GetLastAdjValue()
        {
            int i = this.Count - 1;
            int j = -1;
            while (i >= 0)
            {
                if (this[i] is ValueSE)
                    j = i;
                else
                    if (j != -1)
                        break;
                i--;
            }
            if(j != -1)
                return new SRSingle() { stackIndex = j };
            throw new Exception("No adj value found in the stack");
        }

        public void AddToStack(StackEntry rse)
        {
            int j = this.Count;
            rse.AR = Convert(j % 26);
            this.Add(rse);
        }

        internal string Convert(int index)
        {
            if (index < 0)
                return "?";
            if (index < 26)
            {
                return ((char)('Z' - (index))).ToString();
            }
            else
            {
                return "??";
            }
        }
    }

    internal abstract class StackReference
    {
        
    }

    internal class SRSingle : StackReference
    {
        public int stackIndex;
    }

    internal class SRIterator : StackReference
    {
        public List<SRSingle> refs = new List<SRSingle>();
    }

    internal class VirtualStack : SortedList<int,double>
    {
        
    }
}
