﻿namespace MCalcConsole
{
    internal class Processor
    {
        internal required StackEntriesConsole myStack;
        internal ExtCmdManager myExtCmds = new ExtCmdManager();
        internal Units myUnits = new Units(units_file);
        SortedList<string, CmdProps> mathCmds;
        SortedList<string, CmdPropsML> mathCmdsML;
        SortedSet<string> runningExtCmds = new SortedSet<string>();
        const string ml_type = " :", it_type = " {}";
        const char vc_sep = '|';
        public const string ve_cmd = "@", vc_cmd = "conv", ei_cmd = "in", eo_cmd = "out", units_file = "units_list.txt";

        public Processor()
        {
            mathCmds = new SortedList<string, CmdProps>
            {
                //arithmetic
                { "+", new CmdProps() { cmdfunc = Maths.Sum, minargs = 2, maxargs = int.MaxValue, description = "Addition" } },
                { "++", new CmdProps() { cmdfunc = Maths.Sum, fnum = -1, minargs = 1, maxargs = 1, description = "Add 1" } },
                { "--", new CmdProps() { cmdfunc = Maths.Sum, fnum = -2, minargs = 1, maxargs = 1, description = "Subtract 1" } },
                { "*", new CmdProps() { cmdfunc = Maths.Product, minargs = 2, maxargs = int.MaxValue, description = "Multiplication" } },
                { "|", new CmdProps() { cmdfunc = Maths.Parallel, minargs = 2, maxargs = int.MaxValue, description = "Parallel combination 1/(1/x+1/y)" } },
                { "norm", new CmdProps() { cmdfunc = Maths.Norm, minargs = 2, maxargs = int.MaxValue, description = "Norm (Euclidean distance)" } },
                { "mdist", new CmdProps() { cmdfunc = Maths.Norm, fnum = 1, minargs = 2, maxargs = int.MaxValue, description = "Manhattan distance" } },
                { "-", new CmdProps() { cmdfunc = Maths.Subtract, minargs = 2, maxargs = 2, description = "Subtraction" } },
                { "-'", new CmdProps() { cmdfunc = Maths.Subtract, fnum = 1, minargs = 2, maxargs = 2, description = "Subtraction Y-Z" } },
                { "-<", new CmdProps() { cmdfunc = Maths.Subtract, fnum = 2, minargs = 2, maxargs = 2, description = "Half-difference" } },
                { "-<'", new CmdProps() { cmdfunc = Maths.Subtract, fnum = 3, minargs = 2, maxargs = 2, description = "Half-difference reversed args" } },
                { "-/", new CmdProps() { cmdfunc = Maths.Subtract, fnum = 4, minargs = 2, maxargs = 2, description = "Difference as average ratio" } },
                { "-/'", new CmdProps() { cmdfunc = Maths.Subtract, fnum = 5, minargs = 2, maxargs = 2, description = "Difference as average ratio reversed args" } },
                { "~", new CmdProps() { cmdfunc = Maths.Negate, minargs = 1, maxargs = 1, description = "Negation" } },
                { "abs", new CmdProps() { cmdfunc = Maths.Negate, fnum = 1, minargs = 1, maxargs = 1, description = "Absolute value" } },
                { "/", new CmdProps() { cmdfunc = Maths.Divide, minargs = 2, maxargs = 2, description = "Division" } },
                { "/'", new CmdProps() { cmdfunc = Maths.Divide, fnum = 1, minargs = 2, maxargs = 2, description = "Division Y/Z" } },
                { "\\", new CmdProps() { cmdfunc = Maths.IntPow, fnum = -1, minargs = 1, maxargs = 1, description = "Inversion 1/Z" } },
                { "^", new CmdProps() { cmdfunc = Maths.Pow, minargs = 2, maxargs = 2, description = "Exponentiation" } },
                { "^'", new CmdProps() { cmdfunc = Maths.Pow, fnum = 1, minargs = 2, maxargs = 2, description = "Exponentiation Y^Z" } },
                { "sq", new CmdProps() { cmdfunc = Maths.IntPow, fnum = 2, minargs = 1, maxargs = 1, description = "Square Z*Z" } },
                { "sqrt", new CmdProps() { cmdfunc = Maths.IntRt, fnum = 2, minargs = 1, maxargs = 1, description = "Square root" } },
                { "cb", new CmdProps() { cmdfunc = Maths.IntPow, fnum = 3, minargs = 1, maxargs = 1, description = "Cube" } },
                { "cbrt", new CmdProps() { cmdfunc = Maths.IntRt, fnum = 3, minargs = 1, maxargs = 1, description = "Cube root" } },
                { "rt", new CmdProps() { cmdfunc = Maths.Root, minargs = 2, maxargs = 2, description = "Zth root Y" } },
                { "rt'", new CmdProps() { cmdfunc = Maths.Root, fnum = 1, minargs = 2, maxargs = 2, description = "Yth root Z" } },
                { "pi", new CmdProps() { cmdfunc = Maths.Pi, minargs = 0, maxargs = 0, description = "Pi constant" } },
                { "tau", new CmdProps() { cmdfunc = Maths.Pi, fnum = 1, minargs = 0, maxargs = 0, description = "Tau constant (2*Pi)" } },
                { "dtor", new CmdProps() { cmdfunc = Maths.Pi, fnum = 2, minargs = 1, maxargs = 1, description = "Degrees to radians" } },
                { "rtod", new CmdProps() { cmdfunc = Maths.Pi, fnum = 3, minargs = 1, maxargs = 1, description = "Radians to degrees" } },
                { "ftor", new CmdProps() { cmdfunc = Maths.Pi, fnum = 4, minargs = 1, maxargs = 1, description = "Frequency (1/s) to angular frequency (rad/s)" } },
                { "rtof", new CmdProps() { cmdfunc = Maths.Pi, fnum = 5, minargs = 1, maxargs = 1, description = "Angular frequency (rad/s) to frequency (1/s)" } },
                { ">", new CmdProps() { cmdfunc = Maths.Shift2, fnum = 1, minargs = 1, maxargs = 1, description = "*2" } },
                { ">>", new CmdProps() { cmdfunc = Maths.Shift2, fnum = 2, minargs = 1, maxargs = 1, description = "*4" } },
                { ">>>", new CmdProps() { cmdfunc = Maths.Shift2, fnum = 3, minargs = 1, maxargs = 1, description = "*8" } },
                { ">>>>", new CmdProps() { cmdfunc = Maths.Shift2, fnum = 4, minargs = 1, maxargs = 1, description = "*16" } },
                { "<", new CmdProps() { cmdfunc = Maths.Shift2, fnum = -1, minargs = 1, maxargs = 1, description = "/2" } },
                { "<<", new CmdProps() { cmdfunc = Maths.Shift2, fnum = -2, minargs = 1, maxargs = 1, description = "/4" } },
                { "<<<", new CmdProps() { cmdfunc = Maths.Shift2, fnum = -3, minargs = 1, maxargs = 1, description = "/8" } },
                { "<<<<", new CmdProps() { cmdfunc = Maths.Shift2, fnum = -4, minargs = 1, maxargs = 1, description = "/16" } },
                { "%", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 0, minargs = 2, maxargs = 2, description = "Modulo" } },
                { "%'", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 1, minargs = 2, maxargs = 2, description = "Modulo Y%Z" } },
                { "%i", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 2, minargs = 2, maxargs = 2, description = "IEEE Remainder" } },
                { "%i'", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 3, minargs = 2, maxargs = 2, description = "IEEE Remainder Y%Z" } },
                { "ipart", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 4, minargs = 1, maxargs = 1, description = "Integer part" } },
                { "fpart", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 5, minargs = 1, maxargs = 1, description = "Fractional part" } },
                { "floor", new CmdProps() { cmdfunc = Maths.Round, fnum = -2, minargs = 1, maxargs = 1, description = "Next lower integer" } },
                { "ceil", new CmdProps() { cmdfunc = Maths.Round, fnum = -1, minargs = 1, maxargs = 1, description = "Next higher integer" } },
                { "round", new CmdProps() { cmdfunc = Maths.Round, fnum = 0, minargs = 1, maxargs = 1, description = "Nearest integer" } },
                { "round1", new CmdProps() { cmdfunc = Maths.Round, fnum = 1, minargs = 1, maxargs = 1, description = "Round to 1 significant digit" } },
                { "round2", new CmdProps() { cmdfunc = Maths.Round, fnum = 2, minargs = 1, maxargs = 1, description = "Round to 2 significant digits" } },
                { "round3", new CmdProps() { cmdfunc = Maths.Round, fnum = 3, minargs = 1, maxargs = 1, description = "Round to 3 significant digits" } },
                { "round4", new CmdProps() { cmdfunc = Maths.Round, fnum = 4, minargs = 1, maxargs = 1, description = "Round to 4 significant digits" } },
                { "mean", new CmdProps() { cmdfunc = Maths.Mean, fnum = 0, minargs = 2, maxargs = int.MaxValue, description = "Arithmetic mean" } },
                { "hmean", new CmdProps() { cmdfunc = Maths.Mean, fnum = 1, minargs = 2, maxargs = int.MaxValue, description = "Harmonic mean" } },
                { "gmean", new CmdProps() { cmdfunc = Maths.Mean, fnum = 2, minargs = 2, maxargs = int.MaxValue, description = "Geometric mean" } },
                { "std", new CmdProps() { cmdfunc = Maths.Mean, fnum = 3, minargs = 2, maxargs = int.MaxValue, description = "Standard deviation 1/N" } },
                { "sstd", new CmdProps() { cmdfunc = Maths.Mean, fnum = 4, minargs = 2, maxargs = int.MaxValue, description = "Standard deviation 1/(N-1)" } },
                { "s", new CmdProps() { cmdfunc = Maths.Trig, fnum = 1, minargs = 1, maxargs = 1, description = "Sine" } },
                { "sd", new CmdProps() { cmdfunc = Maths.Trig, fnum = 2, minargs = 1, maxargs = 1, description = "Sine of degrees" } },
                { "c", new CmdProps() { cmdfunc = Maths.Trig, fnum = 3, minargs = 1, maxargs = 1, description = "Cosine" } },
                { "cd", new CmdProps() { cmdfunc = Maths.Trig, fnum = 4, minargs = 1, maxargs = 1, description = "Cosine of degrees" } },
                { "t", new CmdProps() { cmdfunc = Maths.Trig, fnum = 5, minargs = 1, maxargs = 1, description = "Tangent" } },
                { "td", new CmdProps() { cmdfunc = Maths.Trig, fnum = 6, minargs = 1, maxargs = 1, description = "Tangent of degrees" } },
                { "sh", new CmdProps() { cmdfunc = Maths.Trig, fnum = 7, minargs = 1, maxargs = 1, description = "Hyperbolic sine" } },
                { "ch", new CmdProps() { cmdfunc = Maths.Trig, fnum = 8, minargs = 1, maxargs = 1, description = "Hyperbolic cosine" } },
                { "th", new CmdProps() { cmdfunc = Maths.Trig, fnum = 9, minargs = 1, maxargs = 1, description = "Hyperbolic tangent" } },
                { "as", new CmdProps() { cmdfunc = Maths.Trig, fnum = -1, minargs = 1, maxargs = 1, description = "Arcsine" } },
                { "asd", new CmdProps() { cmdfunc = Maths.Trig, fnum = -2, minargs = 1, maxargs = 1, description = "Arcsine as degrees" } },
                { "ac", new CmdProps() { cmdfunc = Maths.Trig, fnum = -3, minargs = 1, maxargs = 1, description = "Arccosine" } },
                { "acd", new CmdProps() { cmdfunc = Maths.Trig, fnum = -4, minargs = 1, maxargs = 1, description = "Arccosine as degrees" } },
                { "at", new CmdProps() { cmdfunc = Maths.Trig, fnum = -5, minargs = 1, maxargs = 2, description = "Arctangent" } },
                { "at2", new CmdProps() { cmdfunc = Maths.Trig, fnum = -5, minargs = 2, maxargs = 2, description = "Arctangent" } },
                { "atd", new CmdProps() { cmdfunc = Maths.Trig, fnum = -6, minargs = 1, maxargs = 2, description = "Arctangent as degrees" } },
                { "atd2", new CmdProps() { cmdfunc = Maths.Trig, fnum = -6, minargs = 2, maxargs = 2, description = "Arctangent as degrees" } },
                { "at2'", new CmdProps() { cmdfunc = Maths.Trig, fnum = -10, minargs = 2, maxargs = 2, description = "Arctangent (args reversed)" } },
                { "at'", new CmdProps() { cmdfunc = Maths.Trig, fnum = -10, minargs = 2, maxargs = 2, description = "Arctangent (args reversed)" } },
                { "atd2'", new CmdProps() { cmdfunc = Maths.Trig, fnum = -11, minargs = 2, maxargs = 2, description = "Arctangent as degrees (args reversed)" } },
                { "atd'", new CmdProps() { cmdfunc = Maths.Trig, fnum = -11, minargs = 2, maxargs = 2, description = "Arctangent as degrees (args reversed)" } },
                { "ash", new CmdProps() { cmdfunc = Maths.Trig, fnum = -7, minargs = 1, maxargs = 1, description = "Hyperbolic arcsine" } },
                { "ach", new CmdProps() { cmdfunc = Maths.Trig, fnum = -8, minargs = 1, maxargs = 1, description = "Hyperbolic arccosine" } },
                { "ath", new CmdProps() { cmdfunc = Maths.Trig, fnum = -9, minargs = 1, maxargs = 1, description = "Hyperbolic arctangent" } },
                { "dp", new CmdProps() { cmdfunc = Maths.VectorProduct, fnum = 0, minargs = 4, maxargs = 4, description = "Dot product of 2 2D vectors" } },
                { "cp", new CmdProps() { cmdfunc = Maths.VectorProduct, fnum = 1, minargs = 4, maxargs = 4, description = "Cross product (determinant) of 2 2D vectors" } },
                { "abv", new CmdProps() { cmdfunc = Maths.VectorProduct, fnum = 2, minargs = 4, maxargs = 4, description = "Counter-clockwise angle between 2 2D vectors" } },
                { "e", new CmdProps() { cmdfunc = Maths.Exp, fnum = -1, minargs = 0, maxargs = 0, description = "e constant" } },
                { "exp", new CmdProps() { cmdfunc = Maths.Exp, fnum = 0, minargs = 1, maxargs = 1, description = "e^Z" } },
                { "exp10", new CmdProps() { cmdfunc = Maths.Exp, fnum = -2, minargs = 1, maxargs = 1, description = "10^Z" } },
                { "exp2", new CmdProps() { cmdfunc = Maths.Exp, fnum = -3, minargs = 1, maxargs = 1, description = "2^Z" } },
                { "ln", new CmdProps() { cmdfunc = Maths.Exp, fnum = 1, minargs = 1, maxargs = 1, description = "Log base e" } },
                { "log10", new CmdProps() { cmdfunc = Maths.Exp, fnum = 2, minargs = 1, maxargs = 1, description = "Log base 10" } },
                { "log2", new CmdProps() { cmdfunc = Maths.Exp, fnum = 4, minargs = 1, maxargs = 1, description = "Log base 2" } },
                { "log", new CmdProps() { cmdfunc = Maths.Exp, fnum = 2, minargs = 1, maxargs = 2, description = "Log base 10 (1 arg) or custom" } },
                { "logb", new CmdProps() { cmdfunc = Maths.Exp, fnum = 2, minargs = 2, maxargs = 2, description = "Log custom base" } },
                { "logb'", new CmdProps() { cmdfunc = Maths.Exp, fnum = 3, minargs = 2, maxargs = 2, description = "Log custom base (args reversed)" } },
                { "log'", new CmdProps() { cmdfunc = Maths.Exp, fnum = 3, minargs = 2, maxargs = 2, description = "Log custom base (args reversed)" } },
            };
            mathCmdsML = new SortedList<string, CmdPropsML>
            {
                { "/%", new CmdPropsML() { cmdfunc = Maths.DivRem, minargs = 2, maxargs = 2, outlines = 2, description = "Division with remainder" } },
                { "/%'", new CmdPropsML() { cmdfunc = Maths.DivRem, fnum = 1, minargs = 2, maxargs = 2, outlines = 2, description = "Division with remainder reversed args" } },
                { "/%i", new CmdPropsML() { cmdfunc = Maths.DivRem, fnum = 2, minargs = 2, maxargs = 2, outlines = 2, description = "Division with IEEE remainder" } },
                { "/%i'", new CmdPropsML() { cmdfunc = Maths.DivRem, fnum = 3, minargs = 2, maxargs = 2, outlines = 2, description = "Division with IEEE remainder reversed args" } },
                { "parts", new CmdPropsML() { cmdfunc = Maths.DivRem, fnum = 4, minargs = 1, maxargs = 1, outlines = 2, description = "Integer and fractional parts" } },
                { "+-", new CmdPropsML() { cmdfunc = Maths.PMT, minargs = 2, maxargs = 2, outlines = 2, description = "Sum and difference" } },
                { "-+", new CmdPropsML() { cmdfunc = Maths.PMT,fnum = -1, minargs = 2, maxargs = 2, outlines = 2, description = "Difference and sum" } },
                { "-+'", new CmdPropsML() { cmdfunc = Maths.PMT,fnum = -2, minargs = 2, maxargs = 2, outlines = 2, description = "Difference and sum reversed args" } },
                { "+-'", new CmdPropsML() { cmdfunc = Maths.PMT, fnum = 1, minargs = 2, maxargs = 2, outlines = 2, description = "Sum and difference reversed args" } },
                { "+-<", new CmdPropsML() { cmdfunc = Maths.PMT, fnum = 2, minargs = 2, maxargs = 2, outlines = 2, description = "Half of Sum and difference" } },
                { "+-<'", new CmdPropsML() { cmdfunc = Maths.PMT, fnum = 3, minargs = 2, maxargs = 2, outlines = 2, description = "Half of Sum and difference reversed args" } },
                { "-+<", new CmdPropsML() { cmdfunc = Maths.PMT,fnum = -3, minargs = 2, maxargs = 2, outlines = 2, description = "Half of Difference and sum" } },
                { "-+<'", new CmdPropsML() { cmdfunc = Maths.PMT,fnum = -4, minargs = 2, maxargs = 2, outlines = 2, description = "Half of Difference and sum reversed args" } },
                { "linf", new CmdPropsML() { cmdfunc = Maths.Linf, minargs = 4, maxargs = 4, outlines = 2, description = "Fit a line y=a*x+b to 2 2D points" } },
                { "rot", new CmdPropsML() { cmdfunc = Maths.Rot2, minargs = 3, maxargs = 3, outlines = 2, description = "Rotate 2D vector counter-clockwise" } },
                { "rotd", new CmdPropsML() { cmdfunc = Maths.Rot2, fnum = 1, minargs = 3, maxargs = 3, outlines = 2, description = "Rotate 2D vector by degrees counter-clockwise" } },
                { "rtop", new CmdPropsML() { cmdfunc = Maths.RadialConv, minargs = 2, maxargs = 2, outlines = 2, description = "Rectangular coordinates to polar coordinates, or (re,im) to (mag,phs)" } },
                { "ptor", new CmdPropsML() { cmdfunc = Maths.RadialConv, fnum = 1, minargs = 2, maxargs = 2, outlines = 2, description = "Polar coordinates to rectangular coordinates, or (mag,phs) to (re,im)" } },
                { "vp", new CmdPropsML() { cmdfunc = Maths.VectorProducts, fnum = 0, minargs = 4, maxargs = 4, outlines = 2, description = "Dot and cross products of 2 2D vectors (cosine and sine of angles between vectors)" } },
                { "sc", new CmdPropsML() { cmdfunc = Maths.Trig2, fnum = 0, minargs = 1, maxargs = 1, outlines = 2, description = "Sine and cosine" } },
                { "scd", new CmdPropsML() { cmdfunc = Maths.Trig2, fnum = 1, minargs = 1, maxargs = 1, outlines = 2, description = "Sine and cosine of degrees" } },
                { "mstd", new CmdPropsML() { cmdfunc = Maths.Stats, fnum = 0, minargs = 2, maxargs = int.MaxValue, outlines = 2, description = "Mean and standard deviation 1/N" } },
                { "msstd", new CmdPropsML() { cmdfunc = Maths.Stats, fnum = 1, minargs = 2, maxargs = int.MaxValue, outlines = 2, description = "Mean and standard deviation 1/(N-1)" } },
                { "box", new CmdPropsML() { cmdfunc = Maths.Stats, fnum = 2, minargs = 2, maxargs = int.MaxValue, outlines = 2, description = "Enclosing range A+-B" } },
            };
        }
        
        public bool ProcessMetaCommand(InputCommand cmdIn, out string? entryText, int atBuffer)
        {
            //check meta commands
            entryText = null;
            int rl;
            switch (cmdIn.cmdword)
            {
                case "b":
                    Program.ClearLine(atBuffer);
                    if (cmdIn.cmdrefs == null || cmdIn.cmdrefs.Count < 1)
                    {
                        rl = myStack.TrimToLength(myStack.Count - 1, atBuffer - 1);
                    }
                    else
                    {
                        if (cmdIn.cmdrefs.Count != 1 || cmdIn.cmdrefs[0] is not SRSingle)
                            throw new Exception("b can use only one reference");
                        SRSingle srs = (SRSingle)cmdIn.cmdrefs[0];
                        rl = myStack.TrimToLength(srs.stackIndex, atBuffer - 1);
                    }
                    rl++;
                    Console.SetCursorPosition(Program.layout_indices[1], rl > 0 ? rl:0);
                    return true;
                case "rs":
                    int nss;
                    if (cmdIn.cmdrefs != null)
                    {
                        if(cmdIn.cmdrefs.Count != 1 || cmdIn.cmdrefs[0] is not SRSingle)
                            throw new Exception("rs can use only one reference");
                        SRSingle srs = (SRSingle)cmdIn.cmdrefs[0];
                        nss = srs.stackIndex;
                    }
                    else
                    {
                        nss = myStack.Count + 1;
                    }
                    RenumberSE rse = new RenumberSE() { TypedEntry = cmdIn.TypedEntry, Comment = cmdIn.Comment, newStart = nss };
                    myStack.AddToStack(rse, atBuffer); //renumber functionality implemented there
                    Console.CursorLeft = Program.layout_indices[1];
                    return true;
                case "cs":
                    if (cmdIn.cmdrefs != null && cmdIn.cmdrefs.Count > 0)
                        throw new Exception("cs cannot contain references");
                    rl = myStack.TrimToLength(0, atBuffer - 1);
                    Console.SetCursorPosition(Program.layout_indices[1], 0);
                    return true;
                case "reload":
                    if (cmdIn.cmdrefs != null && cmdIn.cmdrefs.Count > 0)
                        throw new Exception("reload cannot contain references");
                    myExtCmds = new ExtCmdManager();
                    runningExtCmds.Clear();
                    myUnits = new Units(units_file);
                    Console.CursorLeft = Program.layout_indices[1];
                    return true;
                case "listcmds":
                    if (string.IsNullOrEmpty(cmdIn.Comment))
                        throw new Exception("No file name provided");
                    using (FileStream stream = File.OpenWrite(cmdIn.Comment))
                    {
                        using (StreamWriter sw = new StreamWriter(stream))
                        {
                            foreach (KeyValuePair<string,CmdProps> kv in mathCmds)
                            {
                                sw.Write(kv.Key);
                                sw.Write('\t');
                                sw.Write(kv.Value.minargs);
                                if (kv.Value.maxargs != kv.Value.minargs)
                                {
                                    if (kv.Value.maxargs == int.MaxValue)
                                    {
                                        sw.Write('+');
                                    }
                                    else
                                    {
                                        sw.Write('-');
                                        sw.Write(kv.Value.maxargs);
                                    }
                                }
                                sw.Write('\t');
                                sw.Write(kv.Value.description);
                                sw.WriteLine();
                            }
                            foreach (KeyValuePair<string, CmdPropsML> kv in mathCmdsML)
                            {
                                sw.Write(kv.Key);
                                sw.Write('\t');
                                sw.Write(kv.Value.minargs);
                                if (kv.Value.maxargs != kv.Value.minargs)
                                {
                                    if (kv.Value.maxargs == int.MaxValue)
                                    {
                                        sw.Write('+');
                                    }
                                    else
                                    {
                                        sw.Write('-');
                                        sw.Write(kv.Value.maxargs);
                                    }
                                }
                                sw.Write('\t');
                                sw.Write(kv.Value.outlines);
                                sw.Write('\t');
                                sw.Write(kv.Value.description);
                                sw.WriteLine();
                            }
                        }
                    }
                    Console.CursorLeft = Program.layout_indices[1];
                    return true;
                case "export":
                    if (string.IsNullOrEmpty(cmdIn.Comment))
                        throw new Exception("No file name provided");
                    using(FileStream stream = File.OpenWrite(cmdIn.Comment))
                    {
                        using (StreamWriter sw = new StreamWriter(stream))
                        {
                            foreach (StackEntry se in myStack)
                            {
                                sw.Write(se.TypedEntry);
                                if (se is not ErrorSE)
                                {
                                    sw.Write('\t');
                                    sw.Write(se.Result);
                                }
                                if(se.Comment != null)
                                {
                                    sw.Write('\t');
                                    sw.Write(se.Comment);
                                }
                                sw.WriteLine();
                            }
                        }
                    }
                    Console.CursorLeft = Program.layout_indices[1];
                    return true;
            }
            return false;
        }

        public List<CommandSE> UnpackIterators(InputCommand cmdIn)
        {
            if(cmdIn.cmdword == ve_cmd) //special treatment as multi-line if possible
            {
                if (cmdIn.cmdrefs == null || cmdIn.cmdrefs.Count < 2)
                    throw new Exception(ve_cmd + " requires at least 2 references");
                if(cmdIn.prefs <= 0)
                {
                    cmdIn.prefs = cmdIn.cmdrefs.Count - 1;
                }
                int j = cmdIn.prefs * 2 + 1;
                int k = cmdIn.cmdrefs.Count;
                if (k > j)
                    throw new Exception("Too many input references for command " + ve_cmd);
                j -= k;
                k = 1;
                while (j > 0)
                {
                    cmdIn.cmdrefs.Insert(0, myStack.GetRefByRelativeIndex(k));
                    k++;
                    j--;
                }
                k = cmdIn.cmdrefs.Count - 1;
                cmdIn.prefs = k;
                bool multiAttempt = true;
                for(int i=0; i<k; i++)
                {
                    StackReference sr = cmdIn.cmdrefs[i];
                    if (sr is SRIterator)
                    {
                        int itr = ((SRIterator)sr).refs.Count;
                        if (itr == 0)
                            throw new Exception("Zero length iterator");
                        if (itr == 1)
                        {
                            continue;
                        }
                        else
                        {
                            multiAttempt = false;
                            break;
                        }
                    }
                }
                if (multiAttempt)
                {
                    StackReference sr = cmdIn.cmdrefs[k];
                    if (sr is SRIterator)
                    {
                        int itr = ((SRIterator)sr).refs.Count;
                        if (itr == 0)
                            throw new Exception("Zero length iterator");
                        if (itr > 1)
                        {
                            cmdIn.cmdrefs.RemoveAt(k);
                            cmdIn.cmdrefs.AddRange(((SRIterator)sr).refs);
                        }
                    }
                }
            }
            if (cmdIn.cmdword == vc_cmd) //special treatment to add units in command
            {
                if (string.IsNullOrEmpty(cmdIn.Comment))
                    throw new ArgumentException("Unit conversion string not specified");
                string[] ustr = cmdIn.Comment.Split(' ');
                if (ustr.Length < 2)
                    throw new ArgumentException("Could not parse unit conversion string");
                cmdIn.cmdword = vc_cmd + vc_sep + ustr[0] + vc_sep + ustr[1];
                cmdIn.Comment = string.Join(' ', ustr.Skip(1));
            }
            //check iterators
            if (cmdIn.cmdrefs != null && cmdIn.cmdrefs.Count > 0)
            {
                int itl = 1, itr;
                foreach(StackReference sr in cmdIn.cmdrefs)
                {
                    if(sr is SRIterator)
                    {
                        itr = ((SRIterator)sr).refs.Count;
                        if (itr == 0)
                            throw new Exception("Zero length iterator");
                        if(itr == 1)
                        {
                            continue;
                        }
                        else
                        {
                            if (itl != 1 && itr != itl)
                                throw new Exception("Command iterators are not compatible");
                            itl = itr;
                        }
                    }
                }
                List<CommandSE> iteratedCmds = new List<CommandSE>(itl);
                for(int i=0; i<itl; i++)
                {
                    CommandSE cs = new CommandSE() { cmdword = cmdIn.cmdword, mlin = i, prefs = cmdIn.prefs };
                    List<SRSingle> ls = new List<SRSingle>();
                    foreach (StackReference sr in cmdIn.cmdrefs)
                    {
                        if (sr is SRIterator)
                        {
                            SRIterator si = ((SRIterator)sr);
                            if (si.refs.Count == 1)
                            {
                                ls.Add(si.refs[0]);
                            }
                            else
                            {
                                ls.Add(si.refs[i]);
                            }
                        }
                        else
                        {
                            ls.Add((SRSingle)sr);
                        }
                    }
                    cs.cmdrefs = ls;
                    if (i == 0)
                    {
                        cs.TypedEntry = cmdIn.TypedEntry;
                        cs.Comment = cmdIn.Comment;
                    }
                    else
                    {
                        cs.TypedEntry = it_type;
                        //cs.Comment = cmdIn.Comment;
                    }
                    iteratedCmds.Add(cs);
                }
                return iteratedCmds;
            }

            return new List<CommandSE>() { new CommandSE() { cmdword = cmdIn.cmdword, cmdrefs = null, Comment = cmdIn.Comment, TypedEntry = cmdIn.TypedEntry, prefs = cmdIn.prefs } };
        }

        public List<CommandSE> ProcessCmd(StackEntries sse, CommandSE cmdIn, int i0)
        {
            List<double> rv;            
            if(ProcessLineCommand(sse, i0, ref cmdIn))
            {
                return new List<CommandSE>() { cmdIn };
            }
            if (cmdIn.cmdword == ve_cmd)
            {
                if (cmdIn.cmdrefs == null)
                    throw new Exception(ve_cmd + " (resolved) requires at least 3 references");
                int k = cmdIn.prefs;
                if (k < 2 || ((k % 2) != 0))
                    throw new Exception(ve_cmd + " (resolved) requires 3+2*N references");
                int vs = k / 2;
                int mle = cmdIn.cmdrefs.Count - k;
                VirtualStack newSC = new VirtualStack();
                rv = GetValues(sse, cmdIn.cmdrefs.GetRange(0, vs));
                for (int l = 0; l < vs; l++)
                {
                    int q = cmdIn.cmdrefs[l + vs].stackIndex;
                    double v = rv[l];
                    newSC.Stack.Add(q, new VirtualStack.SCV() { Value = v, Initial = true });
                }
                rv = GetValues(sse, cmdIn.cmdrefs.GetRange(k, mle), ref newSC, 0);
                if (mle == 1)
                {
                    cmdIn.Result = rv[0];
                    return new List<CommandSE>() { cmdIn };
                }
                else
                {
                    List<CommandSE> rl = new List<CommandSE>();
                    for (int l = 0; l < mle; l++)
                    {
                        CommandSE cs = new CommandSE() { cmdword = cmdIn.cmdword, cmdrefs = cmdIn.cmdrefs, mlin = cmdIn.mlin, mlout = l, prefs = cmdIn.prefs };
                        if (l == 0)
                        {
                            cs.TypedEntry = cmdIn.TypedEntry;
                            cs.Comment = cmdIn.Comment;
                        }
                        else
                        {
                            cs.TypedEntry = ml_type;
                        }
                        cs.Result = rv[l];
                        rl.Add(cs);
                    }
                    return rl;
                }
            }
            CmdPropsML? cml;
            if (mathCmdsML.TryGetValue(cmdIn.cmdword, out cml))
            {
                PrependEndRefs(i0, ref cmdIn.cmdrefs, cml.minargs, cml.maxargs);
                rv = GetValues(sse, cmdIn.cmdrefs);
                double[] result = cml.cmdfunc(rv, cml.fnum);
                List<CommandSE> rl = new List<CommandSE>();
                for (int k = 0; k < result.Length; k++)
                {
                    CommandSE cs = new CommandSE() { cmdword = cmdIn.cmdword, cmdrefs = cmdIn.cmdrefs, mlin = cmdIn.mlin, mlout = k, prefs = cmdIn.prefs };
                    if (k == 0)
                    {
                        cs.TypedEntry = cmdIn.TypedEntry;
                        cs.Comment = cmdIn.Comment;
                    }
                    else
                    {
                        cs.TypedEntry = ml_type;
                    }
                    cs.Result = result[k];
                    rl.Add(cs);
                }
                return rl;
            }
            if (myExtCmds.AvailableCommands.Contains(cmdIn.cmdword))
            {
                if (!myExtCmds.LoadedCommands.ContainsKey(cmdIn.cmdword))
                {
                    myExtCmds.LoadCommand(cmdIn.cmdword);
                }
                ExtCmdProps ecp = myExtCmds.LoadedCommands[cmdIn.cmdword];
                PrependEndRefs(i0, ref cmdIn.cmdrefs, ecp.inlines, ecp.inlines);
                rv = GetValues(sse, cmdIn.cmdrefs);
                List<double> result = ProcessExtStack(cmdIn.cmdword, ecp, rv);
                if (result.Count == 1)
                {
                    cmdIn.Result = result[0];
                    return new List<CommandSE>() { cmdIn };
                }
                else
                {
                    List<CommandSE> rl = new List<CommandSE>();
                    for (int l = 0; l < result.Count; l++)
                    {
                        CommandSE cs = new CommandSE() { cmdword = cmdIn.cmdword, cmdrefs = cmdIn.cmdrefs, mlin = cmdIn.mlin, mlout = l, prefs = cmdIn.prefs };
                        if (l == 0)
                        {
                            cs.TypedEntry = cmdIn.TypedEntry;
                            cs.Comment = cmdIn.Comment;
                        }
                        else
                        {
                            cs.TypedEntry = ml_type;
                        }
                        cs.Result = result[l];
                        rl.Add(cs);
                    }
                    return rl;
                }
            }
            throw new Exception("Unknown command type");
        }

        private void PrependEndRefs(int i0, ref List<SRSingle>? rs, int minimumLength, int maximumLength)
        {
            if (rs == null)
            {
                if (minimumLength == 0)
                    return;
                rs = new List<SRSingle>();
            }
            int q = rs.Count;
            if (q > maximumLength)
                throw new Exception("Too many arguments for command");
            if (q >= minimumLength)
                return;
            int j = 1;
            do
            {
                if ((i0 - j) < 0)
                    throw new Exception("Not enough entries in stack");
                rs.Insert(0, new SRSingle() { stackIndex = (i0 - j) });
                j++;
                q++;
            } while (q < minimumLength);
        }

        private List<double> GetValues(StackEntries sse, List<SRSingle>? refs)
        {
            if (refs == null)
                return new List<double>();
            List<double> ol = new List<double>(refs.Count);
            foreach (SRSingle sss in refs)
            {
                StackEntry se = sse.GetByRef(sss);
                if (!se.Result.HasValue)
                    throw new Exception("A referenced entry does not have a numeric value");
                ol.Add(se.Result.Value);
            }
            return ol;
        }

        /*private List<string> GetComments(StackEntries sse, List<SRSingle>? refs)
        {
            if (refs == null)
                return new List<string>();
            List<string> ol = new List<string>(refs.Count);
            foreach (SRSingle sss in refs)
            {
                StackEntry se = sse.GetByRef(sss);
                if (string.IsNullOrEmpty(se.Comment))
                    ol.Add("");
                else
                    ol.Add(se.Comment);
            }
            return ol;
        }*/

        private List<double> GetValues(StackEntries sse, List<SRSingle>? refs, ref VirtualStack stackContext, int rl)
        {
            if (rl >= 10000)
                throw new Exception("Stack recursion error (should never occur)");
            if (refs == null)
                return new List<double>();
            List<double> ol = new List<double>(refs.Count);
            foreach (SRSingle sss in refs)
            {
                VirtualStack.SCV rv;
                double r;
                if (!stackContext.Stack.TryGetValue(sss.stackIndex, out rv))
                {
                    StackEntry se = sse.GetByRef(sss);
                    if (se is ValueSE)
                    {
                        if (!se.Result.HasValue)
                            throw new Exception("A referenced entry does not have a numeric value");
                        r = se.Result.Value;
                    }
                    else
                    {
                        if (se is CommandSE)
                        {
                            if (sss.stackIndex < stackContext.Stack.Keys.First()) //no need to re-evaluate upstream
                            {
                                if (!se.Result.HasValue)
                                    throw new Exception("A referenced entry does not have a numeric value");
                                r = se.Result.Value;
                            }
                            else
                            {
                                
                                ProcessCommandContext(sse, sss, ref stackContext, rl + 1);
                                r = stackContext.Stack[sss.stackIndex].Value;
                            }
                        }
                        else
                        {
                            throw new Exception("A referenced entry is not a command or number");
                        }
                    }
                }
                else
                {
                    r = rv.Value;
                }
                ol.Add(r);
            }
            return ol;
        }

        private bool ProcessLineCommand(StackEntries sse, int i0, ref CommandSE cmd)
        {
            List<double> rv;
            CmdProps? cmdp;
            if (mathCmds.TryGetValue(cmd.cmdword, out cmdp))
            {
                PrependEndRefs(i0, ref cmd.cmdrefs, cmdp.minargs, cmdp.maxargs);
                rv = GetValues(sse, cmd.cmdrefs);
                cmd.Result = cmdp.cmdfunc(rv, cmdp.fnum);
                return true;
            }
            /*if (cmd.cmdword == vd_cmd)
            {
                PrependEndRefs(i0, ref cmd.cmdrefs, 1, 1);
                rv = GetValues(sse, cmd.cmdrefs);
                cmd.Result = rv[0];
                return true;
            }*/
            if (cmd.cmdword.StartsWith(vc_cmd))
            {
                string[] ustr = cmd.cmdword.Split(vc_sep);
                if (ustr[0] == vc_cmd)
                {
                    PrependEndRefs(i0, ref cmd.cmdrefs, 1, 1);
                    rv = GetValues(sse, cmd.cmdrefs);
                    if (ustr.Length < 3)
                        throw new ArgumentException("Could not parse unit conversion string");
                    Unit ui = new Unit(rv[0], ustr[1]);
                    Unit uo = myUnits.ConvertToUnit(ui, ustr[2]);
                    cmd.Result = uo.Multiplier;
                    return true;
                }
            }
            return false;
        }

        private void ProcessCommandContext(StackEntries sse, SRSingle cmd, ref VirtualStack context, int rl)
        {
            StackEntry se = sse.GetByRef(cmd);
            if (se is not CommandSE)
            {
                throw new Exception("Attempt to process command context without command");
            }
            List<double> rv;
            double r;
            double[] ra;
            CmdProps? cmdp;

            CommandSE ncsc = (CommandSE)se;
            //check if multiline, so all output lines may be cached
            int ii = cmd.stackIndex;
            int ij = sse.Count;
            int mls = cmd.stackIndex - ncsc.mlout;
            int mle = ii;
            int lo = ncsc.mlout + 1;
            ii++;
            while (ii < ij)
            {
                if ((sse[ii] is CommandSE) && (((CommandSE)sse[ii]).mlout == lo))
                {
                    mle = ii;
                    lo++;
                }
                else
                {
                    break;
                }
                ii++;
            }

            if (mathCmds.TryGetValue(ncsc.cmdword, out cmdp))
            {
                rv = GetValues(sse, ncsc.cmdrefs, ref context, rl + 1);
                r = cmdp.cmdfunc(rv, cmdp.fnum);
                if (mls != mle)
                    throw new Exception("Multi-line mistmatch (should never see this)");
                context.Stack.Add(cmd.stackIndex, new VirtualStack.SCV() { Value = r, Initial = false });
                return;
            }
            CmdPropsML? cmdpml;
            if (mathCmdsML.TryGetValue(ncsc.cmdword, out cmdpml))
            {
                if ((mle - mls + 1) != cmdpml.outlines)
                    throw new Exception("Multi-line mistmatch (should never see this)");
                rv = GetValues(sse, ncsc.cmdrefs, ref context, rl + 1);
                ra = cmdpml.cmdfunc(rv, cmdpml.fnum);
                if ((mle - mls + 1) != ra.Length)
                    throw new Exception("Multi-line mistmatch 2 (should never see this)");
                for (int i = 0; i < ra.Length; i++)
                {
                    if(!context.Stack.TryAdd(mls + i, new VirtualStack.SCV() { Value = ra[i], Initial = false }))
                    {
                        if (!context.Stack[mls + i].Initial)
                            throw new Exception("Multi-line command overwritten (should never see this)");
                    }
                }
                return;
            }
            /*if (ncsc.cmdword == vd_cmd)
            {
                rv = GetValues(sse, ncsc.cmdrefs, ref context, rl + 1); //gets original values
                if (mls != mle)
                    throw new Exception("Multi-line mistmatch (should never see this)");
                context.Stack.Add(cmd.stackIndex, new VirtualStack.SCV() { Value = rv[0], Initial = false });
                return;
            }*/
            if (ncsc.cmdword.StartsWith(vc_cmd))
            {
                string[] ustr = ncsc.cmdword.Split(vc_sep);
                if (ustr[0] == vc_cmd)
                {
                    rv = GetValues(sse, ncsc.cmdrefs, ref context, rl + 1);
                    if (mls != mle)
                        throw new Exception("Multi-line mistmatch (should never see this)");
                    if (ustr.Length < 3)
                        throw new ArgumentException("Could not parse unit conversion string");
                    Unit ui = new Unit(rv[0], ustr[1]);
                    Unit uo = myUnits.ConvertToUnit(ui, ustr[2]);
                    context.Stack.Add(cmd.stackIndex, new VirtualStack.SCV() { Value = uo.Multiplier, Initial = false });
                    return;
                }
            }
            if (ncsc.cmdword == ve_cmd)
            {
                //treat as multiline if possible
                int k = ncsc.prefs;
                if (k < 2 || ((k % 2) != 0))
                    throw new Exception(ve_cmd + " (resolved 2) requires 3+2*N references");
                int vs = k / 2;
                int mlo = ncsc.cmdrefs.Count - k;
                if ((mlo - 1) != (mle - mls))
                    throw new Exception("Multi-line mistmatch 3 (should never see this)");
                rv = GetValues(sse, ncsc.cmdrefs.GetRange(0, vs), ref context, rl + 1);
                VirtualStack newSC = context.CopyInit();
                for (int i = 0; i < vs; i++)
                {
                    int y = ncsc.cmdrefs[i + vs].stackIndex;
                    double v = rv[i];
                    newSC.Stack[y] = new VirtualStack.SCV() { Value = v, Initial = true };
                }
                rv = GetValues(sse, ncsc.cmdrefs.GetRange(k, mlo), ref newSC, rl + 1);
                for (int i = 0; i < mlo; i++)
                {
                    context.Stack.Add(mls + i, new VirtualStack.SCV() { Value = rv[i], Initial = false });
                }
                return;
            }
            ExtCmdProps? ecmd;
            if (myExtCmds.LoadedCommands.TryGetValue(ncsc.cmdword, out ecmd))
            {
                if ((mle - mls + 1) != ecmd.outlines)
                    throw new Exception("Multi-line mistmatch (should never see this)");
                rv = GetValues(sse, ncsc.cmdrefs, ref context, rl + 1);
                List<double> result = ProcessExtStack(ncsc.cmdword, ecmd, rv);
                if ((mle - mls + 1) != result.Count)
                    throw new Exception("Multi-line mistmatch 2 (should never see this)");
                for (int i = 0; i < result.Count; i++)
                {
                    if (!context.Stack.TryAdd(mls + i, new VirtualStack.SCV() { Value = result[i], Initial = false }))
                    {
                        if (!context.Stack[mls + i].Initial)
                            throw new Exception("Multi-line command overwritten (should never see this)");
                    }
                }
                return;
            }
            throw new Exception("Referenced (substack) entry command could not be interpreted");
        }

        private List<double> ProcessExtStack(string cmname, ExtCmdProps ecp, List<double> inv)
        {
            if (!runningExtCmds.Add(cmname))
                throw new Exception("Circular reference or stall in external command");
            StackEntries ese = ecp.myStack;
            int k = ese.Count;
            int ivi = 0;
            List<double> ouv = new List<double>();
            for(int i=0; i<k; i++)
            {
                StackEntry se = ese[i];
                if(se is CommandSE)
                {
                    CommandSE cse = (CommandSE)se;
                    if(cse.cmdword == ei_cmd)
                    {
                        if (cse.cmdrefs != null && cse.cmdrefs.Count > 0)
                            throw new Exception(ei_cmd + " cannot use references");
                        cse.Result = inv[ivi];
                        ivi++;
                        continue;
                    }
                    if (cse.cmdword == eo_cmd)
                    {
                        PrependEndRefs(i, ref cse.cmdrefs, 1, int.MaxValue);
                        List<double> rv = GetValues(ese, cse.cmdrefs);
                        ouv.AddRange(rv);
                        cse.Result = rv.Last();
                        continue;
                    }
                    List<CommandSE> resl = ProcessCmd(ese, cse, i);
                    int l = resl.Count;
                    if (l < 1)
                        throw new Exception("Unknown return of external command (should never see this)");
                    for(int j=0; j<l; j++)
                    {
                        if (j == 0)
                        {
                            ese[i] = resl[j];
                        }
                        else
                        {
                            if ((i + j) >= k)
                                throw new Exception("Not enough spacer lines to receive multiline outputs in external command");
                            StackEntry sec = ese[i + j];
                            if (sec is CommentSE)
                                ese[i + j] = resl[j];
                            else
                            {
                                if(sec is not CommandSE)
                                    throw new Exception("Incorrect spacer lines to receive multiline outputs in external command");
                                CommandSE rpl = (CommandSE)sec;
                                if (rpl.mlout != j)
                                    throw new Exception("Incorrect spacer lines to receive multiline outputs in external command");
                                ese[i + j] = resl[j];
                            }
                        }
                    }
                    i += (l - 1);
                }
            }
            runningExtCmds.Remove(cmname);
            return ouv;
        }
    }

    internal class CmdProps
    {
        internal delegate double ComputeResult(List<double> args, int fnum);
        required public int minargs, maxargs;
        public int fnum = 0;
        required public ComputeResult cmdfunc;
        public string? description;
    }

    internal class CmdPropsML
    {
        internal delegate double[] ComputeResult(List<double> args, int fnum);
        required public int minargs, maxargs, outlines;
        public int fnum = 0;
        required public ComputeResult cmdfunc;
        public string? description;
    }
}
