﻿namespace MCalcConsole
{
    internal class Maths
    {
        public static double Sum(List<double> args, int fnum) //2+ args
        {
            if (fnum == -1)
                return args[0] + 1;
            if (fnum == -2)
                return args[0] - 1;
            double s = 0;
            foreach(double d in args)
            {
                s += d;
            }
            return s;
        }

        public static double Product(List<double> args, int fnum) //2+ args
        {
            double s = 1;
            foreach (double d in args)
            {
                s *= d;
            }
            return s;
        }

        public static double Norm(List<double> args, int fnum) //1+ args
        {
            double s = 0;
            if (fnum == 1)
            {
                foreach (double d in args)
                {
                    s += double.Abs(d);
                }
                return s;
            }
            if (args.Count == 2)
            {
                return double.Hypot(args[0], args[1]);
            }
            foreach (double d in args)
            {
                s += d * d;
            }
            return double.Sqrt(s);
        }

        public static double Parallel(List<double> args, int fnum)
        {
            double s = 0;
            foreach(double d in args)
            {
                s += (1.0 / d);
            }
            return 1.0 / s;
        }

        public static double Subtract(List<double> args, int fnum) //exactly 2 args
        {
            switch (fnum)
            {
                default:
                    return (args[0] - args[1]);
                case 1:
                    return (args[1] - args[0]);
                case 2:
                    return double.ScaleB(args[0] - args[1], -1);
                case 3:
                    return double.ScaleB(args[1] - args[0], -1);
                case 4:
                    return (args[0] - args[1]) / double.ScaleB(args[0] + args[1], -1);
                case 5:
                    return (args[1] - args[0]) / double.ScaleB(args[1] + args[0], -1);
            }
        }

        public static double Negate(List<double> args, int fnum) //exactly 1 args
        {
            if(fnum == 1)
                return double.Abs(args[0]);
            return -args[0];
        }

        public static double Divide(List<double> args, int fnum) //exactly 2 args
        {
            return fnum == 0 ? (args[0] / args[1]) : (args[1] / args[0]);
        }

        public static double IntPow(List<double> args, int fnum) //exactly 1 args
        {
            return double.Pow(args[0], fnum);
        }

        public static double IntRt(List<double> args, int fnum) //exactly 1 args
        {
            return double.RootN(args[0], fnum);
        }

        public static double Pow(List<double> args, int fnum) //exactly 2 args
        {
            return fnum == 0 ? double.Pow(args[0], args[1]) : double.Pow(args[1], args[0]);
        }

        public static double Exp(List<double> args, int fnum)
        {
            switch (fnum)
            {
                default:
                    return double.Exp(args[0]);
                case -1:
                    return double.E;
                case -2:
                    return double.Exp10(args[0]);
                case -3:
                    return double.Exp2(args[0]);
                case 1:
                    return double.Log(args[0]);
                case 2:
                    if(args.Count < 2)
                        return double.Log10(args[0]);
                    return double.Log(args[0], args[1]);
                case 3:
                    return double.Log(args[1], args[0]);
                case 4:
                    return double.Log2(args[0]);
            }
        }

        public static double Root(List<double> args, int fnum) //exactly 2 args
        {
            if (double.IsInteger(args[1]))
            {
                return fnum == 0 ? double.RootN(args[0], (int)args[1]) : double.RootN(args[1], (int)args[0]);
            }
            return fnum == 0 ? double.Pow(args[0], 1 / args[1]) : double.Pow(args[1], 1 / args[0]);
        }

        public static double Mean(List<double> args, int fnum) //1+ args
        {
            int nn = args.Count;
            double s = 0;
            switch (fnum)
            {
                case 0: //arithmetic
                    foreach (double d in args)
                    {
                        s += d;
                    }
                    return s / nn;
                case 1: //harmonic
                    foreach (double d in args)
                    {
                        s += (1 / d);
                    }
                    return nn / s;
                case 2: //geometric
                    s = 0;
                    foreach (double d in args)
                    {
                        s += double.Log(d);
                    }
                    return double.Exp(s / nn);
                case 3: //standard deviation
                case 4:
                    double s2 = 0, q = args[0], f = 0;
                    foreach (double d in args)
                    {
                        f = d - q;
                        s += f;
                        s2 += (f * f);
                    }
                    if(fnum == 3)
                    {
                        return (s2 - (s * s) / nn) / nn;
                    }
                    return (s2 - (s * s) / nn) / (nn - 1);
            }
            throw new Exception("Incorrect fnum");
        }

        public static double Pi(List<double> args, int fnum) //0 or 1 args
        {
            switch (fnum)
            {
                case 0:
                    return double.Pi;
                case 1:
                    return double.Tau;
                case 2:
                    return args[0] * (double.Pi / 180);
                case 3:
                    return args[0] * (180 / double.Pi);
                case 4:
                    return args[0] * (double.Tau);
                case 5:
                    return args[0] / (double.Tau);
            }
            throw new Exception("Incorrect fnum");
        }

        public static double Shift2(List<double> args, int fnum) //1 args
        {
            return double.ScaleB(args[0], fnum);
        }

        public static double Modulo(List<double> args, int fnum)
        {
            switch (fnum)
            {
                case 0:
                    return args[0] % args[1];
                case 1:
                    return args[1] % args[0];
                case 2:
                    return double.Ieee754Remainder(args[0], args[1]);
                case 3:
                    return double.Ieee754Remainder(args[1], args[0]);
                case 4:
                    return double.Truncate(args[0]);
                case 5:
                    return args[0] % 1;
            }
            throw new Exception("Incorrect fnum");
        }

        public static double Round(List<double> args, int fnum)
        {
            switch (fnum)
            {
                case -2:
                    return double.Floor(args[0]);
                case -1:
                    return double.Ceiling(args[0]);
                case 0:
                    return double.Round(args[0]);
            }
            if(fnum > 0 && fnum < 5)
            {
                if (!double.IsFinite(args[0]) || args[0] == 0)
                    return args[0];
                double a = (args[0] > 0) ? args[0] : -args[0];
                double b = double.Exp10(double.Ceiling(double.Log10(a)) - fnum);
                return double.Round(args[0] / b) * b;
            }
            throw new Exception("Incorrect fnum");
        }

        public static double VectorProduct(List<double> args, int fnum) // x1, y1, x2, y2
        {
            switch (fnum)
            {
                case 0:
                    return (args[0] * args[2]) + (args[1] * args[3]);
                case 1:
                    return (args[0] * args[3]) - (args[1] * args[2]);
                case 2:
                    return Math.Atan2((args[0] * args[3]) - (args[1] * args[2]), (args[0] * args[2]) + (args[1] * args[3]));
            }
            throw new Exception("Incorrect fnum");
        }

        public static double Trig(List<double> args, int fnum)
        {
            switch (fnum)
            {
                case 1:
                    return double.Sin(args[0]);
                case 2:
                    return double.Sin(args[0] * (double.Pi / 180));
                case 3:
                    return double.Cos(args[0]);
                case 4:
                    return double.Cos(args[0] * (double.Pi / 180));
                case 5:
                    return double.Tan(args[0]);
                case 6:
                    return double.Tan(args[0] * (double.Pi / 180));
                case 7:
                    return double.Sinh(args[0]);
                case 8:
                    return double.Cosh(args[0]);
                case 9:
                    return double.Tanh(args[0]);
                case -1:
                    return double.Asin(args[0]);
                case -2:
                    return double.Asin(args[0]) * (180 / double.Pi);
                case -3:
                    return double.Acos(args[0]);
                case -4:
                    return double.Acos(args[0]) * (180 / double.Pi);
                case -5:
                    if (args.Count == 2)
                        return double.Atan2(args[0], args[1]);
                    return double.Atan(args[0]);
                case -6:
                    if (args.Count == 2)
                        return double.Atan2(args[0], args[1]) * (180 / double.Pi);
                    return double.Atan(args[0]) * (180 / double.Pi);
                case -7:
                    return double.Asinh(args[0]);
                case -8:
                    return double.Acosh(args[0]);
                case -9:
                    return double.Atanh(args[0]);
                case -10:
                    if (args.Count == 2)
                        return double.Atan2(args[1], args[0]);
                    return double.Atan(args[0]);
                case -11:
                    if (args.Count == 2)
                        return double.Atan2(args[1], args[0]) * (180 / double.Pi);
                    return double.Atan(args[0]) * (180 / double.Pi);
            }
            throw new Exception("Incorrect fnum");
        }

        public static double[] Linf(List<double> args, int fnum) //(x1,y1,x2,y2) -> (a,b) such that y=a*x+b
        {
            return new double[] { (args[3] - args[1]) / (args[2] - args[0]), (args[1] * args[2] - args[3] * args[0]) / (args[2] - args[0]) };
        }

        /*public static double[] Eqs(List<double> args, int fnum)
        {
            switch (fnum)
            {
                case 1: //(a,b) such that y=a*x+b, return x such that y=0


            }
            return new double[] { (args[3] - args[1]) / (args[2] - args[0]), (args[1] * args[2] - args[3] * args[0]) / (args[2] - args[0]) };
        }*/

        public static double[] DivRem(List<double> args, int fnum) //(x,y)->integer divisor + remainder
        {
            double a = 0, b = 0;
            switch(fnum)
            {
                case 0:
                    a = double.Truncate(args[0] / args[1]);
                    b = args[0] - a * args[1];
                    break;
                case 1:
                    a = double.Truncate(args[1] / args[0]);
                    b = args[1] - a * args[0];
                    break;
                case 2:
                    a = double.Round(args[0] / args[1]);
                    b = args[0] - a * args[1];
                    break;
                case 3:
                    a = double.Round(args[1] / args[0]);
                    b = args[1] - a * args[0];
                    break;
                case 4:
                    a = double.Truncate(args[0]);
                    b = args[0] % 1;
                    break;
                default:
                    throw new Exception("Incorrect fnum");
            }
            return new double[] { a, b };
        }

        public static double[] PMT(List<double> args, int fnum) //exactly 2 args
        {
            double a = 0, b = 0;
            switch (fnum)
            {
                case 0:
                    a = args[0] + args[1];
                    b = args[0] - args[1];
                    break;
                case -1:
                    a = args[0] - args[1];
                    b = args[0] + args[1];
                    break;
                case -2:
                    a = args[1] - args[0];
                    b = args[1] + args[0];
                    break;
                case -3:
                    a = (args[0] - args[1]) / 2;
                    b = (args[0] + args[1]) / 2;
                    break;
                case -4:
                    a = (args[1] - args[0]) / 2;
                    b = (args[1] + args[0]) / 2;
                    break;
                case 1:
                    a = args[1] + args[0];
                    b = args[1] - args[0];
                    break;
                case 2:
                    a = (args[0] + args[1]) / 2;
                    b = (args[0] - args[1]) / 2;
                    break;
                case 3:
                    a = (args[1] + args[0]) / 2;
                    b = (args[1] - args[0]) / 2;
                    break;
                default:
                    throw new Exception("Incorrect fnum");
            }
            return new double[] { a, b };
        }

        public static double[] Rot2(List<double> args, int fnum) // x, y, angle
        {
            double[] r = new double[2];
            double a;
            if (fnum == 1)
                a = args[2] * (double.Pi / 180);
            else
                a = args[2];
            (a, double b) = Math.SinCos(a);
            r[0] = args[0] * b - args[1] * a;
            r[1] = args[0] * a + args[1] * b;
            return r;
        }

        public static double[] RadialConv(List<double> args, int fnum) // x, y <-> R,theta
        {
            double[] r = new double[2];
            if (fnum == 0)
            {
                r[0] = Math.Sqrt((args[0] * args[0]) + (args[1] * args[1]));
                r[1] = Math.Atan2(args[1], args[0]);
            }
            else
            {
                (double a, double b) = Math.SinCos(args[1]);
                r[0] = args[0] * b;
                r[1] = args[0] * a;
            }
            return r;
        }

        public static double[] VectorProducts(List<double> args, int fnum) // x1, y1, x2, y2
        {
            double[] r = new double[2];
            r[0] = (args[0] * args[2]) + (args[1] * args[3]);
            r[1] = (args[0] * args[3]) - (args[1] * args[2]);
            return r;
        }

        public static double[] Trig2(List<double> args, int fnum) // x
        {
            double[] r = new double[2];
            (double,double) q = double.SinCos((fnum <= 0)?args[0]:(args[0]*(double.Pi / 180)));
            if (fnum ==0 || fnum == 1)
            {
                r[0] = q.Item1;
                r[1] = q.Item2;
            }
            else
            {
                r[0] = q.Item2;
                r[1] = q.Item1;
            }
            
            return r;
        }

        public static double[] Stats(List<double> args, int fnum) //1+ args
        {
            int nn = args.Count;
            double s = 0;
            double[] r = new double[2];
            double s2 = 0, q = args[0], f = 0;
            if(fnum == 2)
            {
                s = args[0];
                s2 = args[0];
                foreach (double d in args)
                {
                    if (d < s) s = d;
                    if (d > s2) s2 = d;
                }
                r[0] = (s + s2) / 2;
                r[1] = (s2 - s) / 2;
                return r;
            }
            foreach (double d in args)
            {
                f = d - q;
                s += f;
                s2 += (f * f);
            }
            r[0] = (s / nn) + q;
            if (fnum == 0)
            {
                r[1] = (s2 - (s * s) / nn) / nn;
            }
            else
            {
                r[1] = (s2 - (s * s) / nn) / (nn - 1);
            }
            return r;
        }
    }
}
