//Teensy 4.1

//Pin 13: LED indicator
//Pin 15: input, optical switch, use as QTimer3.3 input
//Pin 41: input, use to test presence of external 5V power
//Pins 18 and 19: I2C communication with INA232 and ADXL345
//Pin 39: output, drive HIGH to enable power to fan motor boost module + fan motor, the motor will start to spin at half-speed or higher depending on motor PWM output pin
//Pin 40: motor sense input, can use as GPT2 Capture2, expect to see two rising edges per revolution of the motor (or four rising+falling edges)
//Pin 33: motor PWM output, drive at 25 kHz to adjust motor speed (from about 50% to 100%)

#include <Wire.h>
#include <DMAChannel.h>

//optomechanics
const uint32_t opt_edges_cirs=8;//power of 2 descriptor of opt_eges_circle
const uint32_t opt_edges_circle=1<<opt_edges_cirs;//measured optical transitions (low to high) per encoder ring revolution
const uint32_t opt_edges_dec=4;//build theta map on every 1<<n edge to reduce computational workload
const uint32_t opt_edges_every=1<<opt_edges_dec;
const uint32_t opt_edges_thetas=opt_edges_circle>>opt_edges_dec;
const uint32_t opt_edges_nbuf=opt_edges_circle*4;//needs to be at least 2*opt_edges_circle and multiple of opt_edges_circle
uint16_t capture_timestamps[opt_edges_nbuf];//buffer to store all captured timestamps for index position identification and theta map building
float thetas[opt_edges_thetas+1], thetas2[opt_edges_thetas+1];//thetas referenced to encoder: (thetas) after filtering, (thetas2) before normalization and filtering
uint32_t capture_ts_write=0, capture_total=0, reference_index=0;
uint32_t err_slow=0, err_lock=0;
elapsedMicros opt1_el, opt2_el;//Note: elapsedMicros can be used in interrupts, elapsedMillis can not
uint32_t opt1_el_min=1000000, opt1_el_max=0, opt1_dur_min=1000000, opt1_dur_max=0, opt2_el_min=1000000, opt2_el_max=0, opt2_dur_min=1000000, opt2_dur_max=0;
volatile uint32_t ld_sync_var=0;//valid when theta observer operational, upper 16 bits = timestamp at specified position, next 4 bits = reduced theta position reference of timestamp, least 12 bits = timer ticks per reduced theta step
volatile float theta_est, omega_est; //theta in normalized angle [0,1] from reference position at last optical interrupt (valid when observer operational), omega in timer ticks per revolution (inverse of velocity) valid in all observer states
const uint32_t timer_buffer_bytes=64;//(16 thetas*2 pixels per theta)*2 bytes per pixel
__attribute__((aligned(timer_buffer_bytes))) uint16_t timer_buffer[timer_buffer_bytes/2];//length of buffer that can be filled with known ld_sync_var, to reduce interrupt load, since this is not updated at every optical edge
DMAChannel tmrWriter;//this continuously loads the next pixel timer compare values from timer_buffer into TMR3_COMP10 for the next update, triggered by TMR3.0 compare event
volatile byte opt_state=0; //optical tracker state machine 0 = starting up, 15 = theta observer operational
uint8_t tmr3_slow_div=12;//divide TMR3.3 clock of (150MHz/16) further by (this+1) to slow down TMR3.0 and 3.1 so that 1 revolution takes less than 65536 ticks, to unambiguously identify theta
  //with 12, this gives (150/16)/13=721153.85 Hz, which gives 65536 counts in 1/11 of a second (usable at >11 RPS)
//int theta_alpha_N=49, omega_alpha_N=-49;//linearized scale for IIR coefficients, higher number >0 = more samples averaged in IIR, 0=equal contribution of new sample and avg, lower number <0 = reduced contribution of average
float theta_alpha=0.99, omega_alpha=0.01;//IIR coefficients for angular position and velocity, between 0 and 1, high value = averaging is preferred, low value = measurement is preferred

//pixels
volatile uint16_t target_pixel;//target pixel for timing calculation in timer loading DMA interrupt (this is ahead of FlexIO2 DMA)
volatile int north_pixel_shift=0;//used to shift the pixels in theta when parsing pixel buffer, independent of theta shift which is linked to mechanical location of encoder
volatile float theta_pixel_shift=-160.6;//-256 to +256 (allowed -511 to +511), shift of angle (in pixels) from theta referenced to encoder, to pixel reference frame
volatile float theta_pixel_vshift=0;//-751 to 751 additional shift that is velocity-dependent due to pixel setup time, as theta_pixel_vshift/(omega_est/(2*opt_edges_every)) (note this should be smaller than 1 pixel with timer durations up to 65536 ticks per revolution)
  //nominally with 11.67 us pixel offset this is 269.2 = 350 cycles/30 MHz FlexIO clock*721153.85 Hz TMR3 clock*32 pixels per coarse theta step, the +-751 limit corresponds to a full 32*pixel step at 30 Hz
uint32_t pixel_buffer[4096];//512*128*2/32 each pixel is 2 bits
  //buffer is organized such that column 0+ corresponds to negative theta and reaches positive theta (division is such that theta=0 at c=255.5, theta=+-180deg at c=-0.5=511.5)
  //  theta is increasing in positive direction counter-clockwise, which is also the direction of mirror spin
  //row 0+ corresponds to zenith going down to horizon in phi (division is such that zenith is at -0.5 and horizon at 127.5)
  //given c between 0 and 511, and r between 0 and 127,
  //  the pixel bits at given location are: (LSBs correspond to lower columns)
  //  (pixel_buffer[r*32+(c/16)]>>((c%16)*2))&0x03
const uint32_t laser_buffer_bytes=49152;//8*512*3*4 each FlexIO shift is 8 uint32_t, 512 pixels, and there are 3 shifts per pixel, 4 bytes per uint32_t
__attribute__((aligned(64))) uint32_t laser_buffer[laser_buffer_bytes/4];//12288
  //defined such that at index 0 the data is defined for first positive theta pixel for LD0; at theta=0 the mirror is perpendicular to LD0 (this is not a pixel spot though)
  //LD0 is the most-clockwise LD on the board connected to the _1 port (Shifter0 or bus A)
  //  which corresponds to the first data bit shifted out (LSB of shiftbuf, or MSB of shiftbufbis)
  //  and going up to LD31 for the last data bit shifted out
  //  and LD32-63 on board _2 (Shifter1 or bus B) and so on
DMAChannel ldWriter;//this continuously loads the next values from laser_buffer into 8 FlexIO shift buffer registers for the next update, triggered by FlexIO shift buffer flag
volatile uint32_t test_ld=210, test_ld_mode=0;
volatile byte bright_mode=0;//change with setBrightMode() to 2 for dynamic range of 10.4 (2,6.8,20.8)us, 1 for dynamic range of 7 (2,4.8,14)us, and 0 for dynamic range of 42 (0.33,2,14)us
bool ds_or_mode=true;//dots "or" mode, if true then pixel buffer gets or-ed while writing, if false it is overwritten by latest dot
uint8_t pxt0lsb=71;//(this+1)*2 30 Mhz cycles between pixel step triggers, minimum 65 (132 cycles) for complete 32-bit shift+latch
uint16_t pxt1dly=283, pxt2dly=211, pxt5dly=129, pxt7dly=625;//(this+1) 30 MHz cycles from start of transmission to emitting a pulse: 1-reset, 2-enable, 5-latch (min 129 (130 cycles) for 32 bits), 7-disable

//accessories
//i2c addresses
const byte INA232_addr=0x40;
const byte ADXL345_addr=0x53;
uint32_t c[14];
float siw[opt_edges_thetas], cow[opt_edges_thetas];//sine and cosine waves for accelerometer phase determination
uint32_t adxl_ovr_errs=0;
bool adxl_running=false;
uint32_t adxl_samples_acquired=0, adxl_num_avg=1638, adxl_samples[opt_edges_thetas];
uint32_t zrmssq=0, zrmsnum=0; int32_t zrmslin=0, zrmsref=0; //RMS number of samples should not exceed 16384 to avoid overflow of these
float vibx[opt_edges_thetas],viby[opt_edges_thetas],vibz[opt_edges_thetas],vibr[opt_edges_thetas];
elapsedMicros el_adxl;
uint32_t t_adxl_min=1000000, t_adxl_max=0;
volatile bool display_active=false;
byte motor_default_speed=0, motor_max_speed=255;
bool motor_auto_start=true, motor_auto_stop=true, display_auto_start=true, eth_led_enable=false;
uint32_t disconnect_stopdisplay=100000; // [ms] if display is active, and there is no ethernet connection for 100 seconds, stop display; set 0 to not stop; limit of 255000 by eeprom byte use
uint32_t motor_spinup_time=10000; //[ms] while spinning up motor and starting theta observer, if no lock is achieved in this time the motor is shut down
uint32_t cmd_burst_interval=10; // [ms] if commands arrive with less than this many ms between them, then consider that we have lost synchronization somehow and close the connection; limit of 255 by eeprom byte use
uint32_t display_active_time=0;//ms for informational display on webpage
elapsedMillis disconnect_timer=0, burst_timer=0, display_timer=0;

//ethernet
#include <eeprom.c>
#include <algorithm>
#include <cstdio>
#include <utility>
#include <vector>
#include <QNEthernet.h> // at C:\Users\Max\Documents\Arduino\libraries\QNEthernet\src\util\ip_tools.cpp

using namespace qindesign::network;

// --------------------------------------------------------------------------
//  Ethernet Configuration
// --------------------------------------------------------------------------
// The DHCP timeout, in milliseconds. Set to zero to not wait and
// instead rely on the listener to inform us of an address assignment.
constexpr uint32_t kDHCPTimeout = 15'000;  // 15 seconds
// The link timeout, in milliseconds. Set to zero to not wait and
// instead rely on the listener to inform us of a link.
constexpr uint32_t kLinkTimeout = 5'000;  // 5 seconds
uint16_t kServerPort = 80; // may be modified by changing EEPROM
// Timeout for waiting for input from the client.
constexpr uint32_t kClientTimeout = 5'000;  // 5 seconds
// Timeout for waiting for a close from the client after a
// half close.
constexpr uint32_t kShutdownTimeout = 30'000;  // 30 seconds
// Set the static IP to something other than INADDR_NONE (all zeros)
// to not use DHCP. The values here are just examples.
IPAddress staticIP{192, 168, 1, 110}; // may be modified by changing EEPROM
IPAddress subnetMask{255, 255, 255, 0};
IPAddress gateway{192, 168, 1, 1};
// Keeps track of state for a single client.
const int cinbuflen=120;
uint32_t ethBytesReceived=0, ethDataMaxTime=0, pixGenMaxTime=0;//for display purposes on webpage
elapsedMicros ethProcessTime;
struct ClientState {
  ClientState(EthernetClient client)
      : client(std::move(client)) {}

  EthernetClient client;
  bool closed = false;

  // For timeouts.
  uint32_t lastRead = millis();  // Mark creation time
  bool keepopen = false;

  // For half closed connections, after "Connection: close" was sent
  // and closeOutput() was called
  uint32_t closedTime = 0;    // When the output was shut down
  bool outputClosed = false;  // Whether the output was shut down

  // Parsing state
  bool emptyLine = false;
  byte parsingState=0;
  int bytesToReceive=0, bytesReceived=0, inctr=0;
  uint32_t prevAddr=0, upperAddr=0;
  char instr[cinbuflen];
};
// Keeps track of what and where belong to whom.
std::vector<ClientState> clients;
// The server.
EthernetServer server{kServerPort};
const uint32_t firmware_buffer_length=300000;//bytes reserved for receiving new firmware image over internet
const uint32_t flash_offset=0x60000000;//address from which to write to/from buffer
__attribute__((aligned(64))) DMAMEM byte firmware_buffer[firmware_buffer_length];

void setup() {
  Wire.begin();//Pins 18 and 19: I2C communication with INA232 and ADXL345
  Wire.setClock(400000);
  Serial.begin(115200);
  pinMode(13, OUTPUT); //LED indicator
  pinMode(15, INPUT); //optical switch
  pinMode(41, INPUT); //external 5V status
  pinMode(39, OUTPUT); //motor power enable
  digitalWrite(39, LOW); //disable motor
  pinMode(33, OUTPUT); //motor PWM
  digitalWrite(33, LOW); //low output
  //analogWriteFrequency(33, 25000); //set up PWM output at 25 kHz
  pinMode(40, INPUT); //motor sense status

  pinMode(0, OUTPUT);//Debug indicator (NC)
  pinMode(1, OUTPUT);//Debug indicator (NC)
  pinMode(2, OUTPUT);//Debug indicator (NC)
  pinMode(3, OUTPUT);//Debug indicator (NC)

  //DMAPriorityOrder(ldWriter, tmrWriter);//ldWriter has to complete more quickly than tmrWriter; this needs to be called before setting up the DMA channels

  for(int i=0; i<opt_edges_thetas; i++){
    siw[i]=sin((6.2831853f/opt_edges_thetas)*i);
    cow[i]=cos((6.2831853f/opt_edges_thetas)*i);
  }
  
  ina232_start(INA232_addr);

  loadFromEeprom();//get previously saved values

  if(digitalRead(41)){
    //initialize hardware only if external 5V power is available
    setupQTimer();
    setupFlexIO2();
  }else{
    Serial.println("5V not available - safe mode");
  }

  setupEthernet();
}

void loop() {
  processSerial();
  processEthernetClients();
  //delay(1);
  if(adxl_running){
    processADXL();
  }
  yield();//process ethernet functions and others
  //if(p13_led_enable){ digitalWrite(13,!digitalRead(13)); }
}
