﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Automation;
using static AstroDisplayUI.SolarSystemCalc;
using System.Windows.Controls;
using System.Windows.Documents;

namespace AstroDisplayUI
{
    internal class StarsCalc
    {
        public SortedList<int,StarInfo> Stars = new SortedList<int,StarInfo>();
        public SortedList<string,ConstellationFigure> Constellations = new SortedList<string,ConstellationFigure>();


        public AstroPosition CalculateStarPosition(StarInfo si, DateTime dobs, double obs_latitude_degn, double obs_longitude_dege)
        {
            //dobs = local date and time of observation
            //obs_latitude_degn = observation latitude in degrees north (-90 to 90, 39 for Washington DC), for azimuth and altitude
            //obs_longitude_dege = observation longitude in degrees east (-180 to 180, 77 for Washington DC), for approximate UTC correction, not too important

            DateTime dobs_utc = dobs - TimeSpan.FromDays(obs_longitude_dege / 360);//approximate UTC date of observation
            //the above conversion does not take into account time zones or equation of time
            //since this is only used to calculate orbit parameters which change slowly
            //there is a DateTime.ToUniversalTime() function but it uses an implicit time zone
            DateTime d0 = new DateTime(1999, 12, 31);
            //day number
            TimeSpan dayNum = dobs_utc - d0;
            double d = dayNum.TotalDays; //[days] since Dec 31 1999 UST (approximate) used for orbital parameters
            double ecl = (23.4393 - (3.563e-7 * d) % 360) * DTOR;//[rad] obliquity of the ecliptic
            OrbitalElements Sun = new OrbitalElements(new double[] { 0, 0, 0, 0, 282.9404, 4.70935e-5, 1, 0, 0.016709, -1.151e-9, 356.0470, 0.9856002585 }, d);

            //convert further to local coordinates
            double obs_latitude = obs_latitude_degn * DTOR; //[rad]
            double obs_localtime = dobs.TimeOfDay.TotalHours * (Math.PI / 12); //[rad] 0 to 2*pi, pi = noon
            //the above local time does not take into account time zones or equation of time
            //therefore the result may correspond to almanac entries up to ~30 minutes earlier or later than the time specified in dobs
            double Ls = Sun.w + Sun.M; //mean longitude
            double gmst0 = Ls + Math.PI;
            double sidtime = obs_localtime + gmst0;

            AstroPosition StarEcl = new AstroPosition() { C1 = si.elon * DTOR, C2 = si.elat * DTOR, Radius = 1 };
            AstroPosition StarEq = AstroPosition.EclRotate(StarEcl, ecl);
            AstroPosition StarAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(StarEq, sidtime), Math.PI / 2 - obs_latitude);

            return StarAz;
        }

        public void ParseStarsFile()
        {
            SortedList<int, string> StarNames = ParseStarNames();

            string fn = "V53A Stars.txt";
            Stars.Clear();
            int state = 0;
            using (FileStream fs = File.OpenRead(fn))
            {
                using (StreamReader sr = new StreamReader(fs))
                {
                    string l;                    
                    while(!sr.EndOfStream)
                    {
                        l=sr.ReadLine();
                        if (l == null)
                            break;
                        if (l.StartsWith("----"))
                        {
                            state = state + 1;
                            if (state == 3)
                                break;
                            continue;
                        }
                        if (state != 2)
                        {
                            continue;
                        }
                        //process star descriptor line
                        string[] la = l.Split('|');
                        int hr = int.Parse(la[0].Trim());
                        string nm;
                        if (StarNames.ContainsKey(hr))
                        {
                            nm = StarNames[hr];
                        }
                        else
                        {
                            nm = null;
                        }
                        StarInfo si = new StarInfo()
                        {
                            hip = int.Parse(la[12].Trim()),
                            elon = float.Parse(la[9].Substring(0, 6).Trim()),
                            elat = float.Parse(la[9].Substring(7).Trim()),
                            mag = float.Parse(la[22].Trim()),
                            cnst = la[4].Trim(),
                            bayr = la[3].Trim(),
                            name = nm
                        };
                        try
                        {
                            Stars.Add(si.hip, si);
                        }catch(ArgumentException x) { }
                    }
                }
            }
            if (state != 3)
            {
                throw new Exception("Parse star details error");
            }
        }

        SortedList<int, string> ParseStarNames()
        {
            SortedList<int, string> StarNames = new SortedList<int, string>();
            
            string fn = "V53A Star Names.txt";
            int state = 0;
            using (FileStream fs = File.OpenRead(fn))
            {
                using (StreamReader sr = new StreamReader(fs))
                {
                    string l;
                    while (!sr.EndOfStream)
                    {
                        l = sr.ReadLine();
                        if (l == null)
                            break;
                        if (l.StartsWith("----"))
                        {
                            state = state + 1;
                            if (state >= 3)
                                break;
                            continue;
                        }
                        if (state != 2)
                        {
                            continue;
                        }
                        //process star name line
                        string[] la = l.Split('|');
                        try
                        {
                            StarNames.Add(int.Parse(la[0].Trim()), la[1]);
                        }catch(ArgumentException x)
                        {
                            //skip
                        }
                    }
                }
            }
            if(state != 3)
            {
                throw new Exception("Parse star names error");
            }
            return StarNames;
        }

        public void ParseConstellationsFile()
        {
            Constellations = new SortedList<string,ConstellationFigure>();
            string fn = "constellations_stellarium.txt";
            int state = 0, cline = 0;
            ConstellationFigure cn = null;
            using (FileStream fs = File.OpenRead(fn))
            {
                using (StreamReader sr = new StreamReader(fs))
                {
                    string l, c = null, g = null;
                    while (!sr.EndOfStream)
                    {
                        l = sr.ReadLine();
                        if (l == null)
                            break;
                        if (l.StartsWith("----"))
                        {
                            state = state + 1;
                            if (state >= 3)
                            {
                                if (cn != null)
                                {
                                    Constellations.Add(cn.name, cn);
                                    cn = null;
                                }
                                break;
                            }
                            continue;
                        }
                        if (state != 2)
                        {
                            continue;
                        }
                        //process constellation line
                        if (string.IsNullOrWhiteSpace(l))
                        {
                            if (cn != null)
                            {
                                Constellations.Add(cn.name, cn);
                                cn = null;
                            }
                            cline = 0;
                            continue;
                        }
                        if (cline < 1)
                        {
                            int ls = l.IndexOf('|'); //description separator
                            if (ls == -1)
                            {
                                cn = new ConstellationFigure() { name = l.Trim(), lines = new List<ConstellationFigure.CFLine>() };
                            }
                            else
                            {
                                cn = new ConstellationFigure() { name = l.Substring(0,ls).Trim(), desc = l.Substring(ls+1).Trim(), lines = new List<ConstellationFigure.CFLine>() };
                            }
                            cline = 1;
                        }
                        else
                        {
                            string[] sl = l.Split('-');
                            if (sl.Length < 2)
                                throw new Exception("Constellation line segment must contain 2 or more entries");
                            int hip0 = -1, hip1 = -1;
                            foreach (string ss in sl)
                            {
                                string st = ss.Trim();
                                if (st[0]>='0' && st[0]<='9')
                                {
                                    //treat as HIP number
                                    hip1 = int.Parse(st);
                                    if (!Stars.ContainsKey(hip1))
                                        hip1 = -1;
                                        //throw new Exception("Star HIP number from constellation could not be found in star catalog");
                                }
                                else
                                {
                                    //treat as constellation and greek letter name
                                    int spi = st.IndexOf(' ');
                                    if (spi >= 0)
                                    {
                                        c = st.Substring(0, spi);
                                        g = st.Substring(spi + 1).Trim();
                                    }
                                    else
                                    {
                                        g = st;
                                    }
                                    if (string.IsNullOrWhiteSpace(c) || string.IsNullOrWhiteSpace(g))
                                        throw new Exception("Incorrect definition of star letter identifier");
                                    try
                                    {
                                        KeyValuePair<int, StarInfo> sip = Stars.First(p => (p.Value.cnst == c && p.Value.bayr == g));
                                        hip1 = sip.Key;
                                    }
                                    catch (Exception e)
                                    {
                                        throw new Exception("Unable to find star in catalog by constellation name and letter", e);
                                    }
                                }
                                if (hip0 >= 0 && hip1 >= 0)
                                {
                                    cn.lines.Add(new ConstellationFigure.CFLine(hip0, hip1));
                                }
                                hip0 = hip1;
                            }
                        }
                    }
                }
            }
            if (state != 3)
            {
                throw new Exception("Parse star names error");
            }
            //return constellations;
        }

        internal class StarInfo
        {
            public int hip;
            public double elon, elat;
            public float mag;
            public string name, cnst, bayr;
            public override string ToString()
            {
                return cnst + " " + bayr + " " + name;
            }
        }

        internal class ConstellationFigure
        {
            public string name;
            public string desc;
            public struct CFLine
            {
                public int S1, S2;
                public CFLine(int s1, int s2)
                {
                    S1 = s1;
                    S2 = s2;
                }
            }
            public List<CFLine> lines;
            public override string ToString()
            {
                return name;
            }
        }
    }
}
