﻿using System;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading.Tasks;
using System.Windows;

namespace AstroDisplayUI
{
    internal class LaserProjector
    {
        public bool IsConnected { get { return _isConnected; } }
        bool _isConnected = false;
        Socket device;
        IPAddress lpip = IPAddress.Parse("192.168.1.110");
        byte[] screenBuffer = new byte[16384], dotBuffer = new byte[16384];
        int dotsused = 0;
        bool usingScreenBuffer = false;
        int phiRowLimit = 3;//bottom N rows are not drawn
        const uint connection_response_code = 0x58AD4166;//received upon opening connection
        const uint flash_write_code = 0x1AE022D8;//code to start writing data to flash
        string connection_request = "ASTROy\r\n", flash_request = "ASTROf\r\n";

        ~LaserProjector()
        {
            if (device != null)
            {
                device.Shutdown(SocketShutdown.Both);
                device.Close();
            }
        }

        public async Task Connect()
        {
            if (device == null || !_isConnected)
            {
                try
                {
                    device = new Socket(SocketType.Stream, ProtocolType.Tcp);
                    //device.SendBufferSize = 16400;
                    //device.SendTimeout = 2000;
                    //device.ReceiveTimeout = 2000;
                    device.Connect(lpip, 80);
                    device.Send(Encoding.ASCII.GetBytes(connection_request));
                    uint rc = 0;
                    await Task.Run(() => { rc = receiveUint(); });
                    if (rc == connection_response_code)
                    {
                        _isConnected = true;
                    }
                    else
                    {
                        _isConnected = false;
                        device.Disconnect(false);
                    }
                    //device = new SerialPort("COM3", 115200);
                    //device.Open();
                    //device.Write("S");//start operation
                }
                catch (Exception x)
                {
                    _isConnected = false;
                    device = null;
                }
            }
        }
        
        public void Disconnect()
        {
            if (device != null && _isConnected)
            {
                //device.Write("p");//stop operation
                //device.Close();
                device.Shutdown(SocketShutdown.Both);
                device.Close();
                device = null;
            }
            _isConnected = false;
        }

        public void DrawPoint(double theta, double phi, byte brightness)
        {
            //this function takes theta in radians from north (0) towards west (+), phi in radians from zenith (0) down to horizon (+)
            theta = Math.IEEERemainder(theta, 2*Math.PI);//ensure within +-Pi
            double thetapx = Math.Round(theta * (256 / Math.PI) + 256);//convert to pixel coords
            if (thetapx < 0)
                thetapx += 512;
            if (thetapx >= 512)
                thetapx -= 512;
            phi = Math.IEEERemainder(phi, Math.PI);//ensure within +-Pi/2
            double phipx = Math.Round(phi * (256 / Math.PI));//convert to pixel coords
            if(phipx < 0)
            {
                phipx = -phipx;
                thetapx = thetapx + 256;
                if (thetapx >= 512)
                    thetapx -= 512;
            }
            if (phipx > (128 - phiRowLimit))
                return; //too low below horizon
            addPoint((uint)thetapx, (uint)phipx, brightness);
        }

        public bool SendData()
        {
            if (!_isConnected || device==null)
                return false;
            int bytes;
            if (!usingScreenBuffer)
            {
                int bits = dotsused * 18;
                bytes = bits / 8;
                if (bits % 8 != 0)
                    bytes++;
                sendUint((uint)bytes);
                sendArray(dotBuffer, bytes);
            }
            else
            {
                bytes = 16384;
                sendUint((uint)bytes);
                device.Send(screenBuffer, bytes, SocketFlags.None);
            }
            if (receiveUint() == (uint)bytes)
            {
                //confirmation received
                return true;
            }
            else
            {
                //something wrong
                return false;
            }
        }

        public void SetBrightMode(byte brt)
        {
            if (_isConnected)
            {
                if (brt==1)
                {
                    sendUint(65541);
                }
                else
                {
                    if (brt == 2)
                    {
                        sendUint(65542);
                    }
                    else
                    {
                        sendUint(65540);
                    }
                }
                uint res=receiveUint();//should equal command
            }
        }

        public void StartDisplay(bool on)
        {
            if (_isConnected)
            {
                if (on)
                {
                    sendUint(65537);
                }
                else
                {
                    sendUint(65536);
                }
                uint res = receiveUint();//should equal command
            }
        }

        public double GetMotorSpeed()
        {
            if (!_isConnected) { return double.NaN; }
            sendUint(131111);
            return receiveUint() * 1e-3; // [RPS]
        }

        public double GetSupplyVoltage()
        {
            if (!_isConnected) { return double.NaN; }
            sendUint(131100);
            return receiveUint() * 1e-3; // [V]
        }

        public double GetSupplyCurrent()
        {
            if (!_isConnected) { return double.NaN; }
            sendUint(131101);
            return receiveUint() * 1e-3; // [A]
        }

        void sendArray(byte[] arr, int len)
        {
            /*int os = 0;
            int ls = len;
            const int chunk = 100;
            while (ls > 0)
            {
                int bs = device.Send(arr, os, ls>chunk? chunk:ls, SocketFlags.None);
                os += bs;
                ls -= bs;
            }*/
            device.Send(arr, len, SocketFlags.None);
        }

        void sendUint(uint bd)
        {
            byte[] bs = { 0, 0, 0, 0 };
            for (int q = 0; q < 4; q++)
            {
                bs[q] = (byte)((bd & 0xFF000000) >> 24);
                bd <<= 8;
            }
            device.Send(bs);
        }

        uint receiveUint()
        {
            byte[] buf = { 0, 0, 0, 0 };
            device.Receive(buf);
            uint bd = 0;
            for (int q = 0; q < 4; q++)
            {
                bd <<= 8;
                bd |= buf[q];
            }
            return bd;
        }

        public void OutlinePoint(double theta, double phi, byte brightness, int level)
        {
            for(int a=-level; a<=level; a++)
            {
                for (int b = -level; b <= level; b++)
                {
                    DrawPoint(theta + a * (Math.PI / 256), phi + b * (Math.PI / 256), brightness);
                }
            }
        }

        public void DrawGrid()
        {
            ClearBuffer();
            prepareScreenBuffer();
            for (uint r = 1; r < 9; r++) //latitude lines
            {
                double phi = r * 10;//0 at zenith
                uint p = (uint)Math.Round((phi - (90.0 / 256)) / 90.0 * 128);
                byte b = (byte)((r % 3 == 0) ? 2 : 1);
                for (uint t = 0; t < 512; t++)
                {
                    addPoint(t, p, b);
                }
            }
            for (uint c = 0; c < 36; c++) //longitude lines
            {
                double theta = c * 10;//0 at north
                uint t = (uint)Math.Round((theta + 180) / 360.0 * 512);
                byte b = (byte)((c % 3 == 0) ? 2 : 1);
                for (uint p = 0; p < (128-phiRowLimit); p++)
                {
                    addPoint(t, p, b);
                }
            }
            string N =
                ".   ." +
                ".   ." +
                "..  ." +
                ". . ." +
                ".  .." +
                ".   ." +
                ".   .";
            string E =
                "....." +
                ".    " +
                ".    " +
                ".... " +
                ".    " +
                ".    " +
                ".....";
            string S =
                " ...." +
                ".    " +
                ".    " +
                " ... " +
                "    ." +
                "    ." +
                ".... ";
            string W =
                ".   ." +
                ".   ." +
                ".   ." +
                ". . ." +
                ". . ." +
                ". . ." +
                " . . ";
            drawSymbol(N, 256 + 2, (uint)(128 - phiRowLimit - 10));
            drawSymbol(E, 128 + 2, (uint)(128 - phiRowLimit - 10));
            drawSymbol(S, 0 + 2, (uint)(128 - phiRowLimit - 10));
            drawSymbol(W, 384 + 2, (uint)(128 - phiRowLimit - 10));
        }

        private void drawSymbol(string s, uint t_offset, uint p_offset)
        {
            for (uint c = 0; c < 5; c++)
            {
                for (uint r = 0; r < 7; r++)
                {
                    if (s[(int)(r * 5 + c)] == '.')
                    {
                        addPoint(t_offset - c, p_offset + r, 3);
                    }
                }
            }
        }

        public void DrawLine(double theta1, double phi1, double theta2, double phi2, byte brightness)
        {
            double dtheta = Math.IEEERemainder(theta2 - theta1, Math.PI * 2);
            double dphi = Math.IEEERemainder(phi2 - phi1, Math.PI);
            int tsteps = (int)Math.Ceiling(Math.Abs(dtheta) / (Math.PI / 256));
            int psteps = (int)Math.Ceiling(Math.Abs(dphi) / (Math.PI / 256));
            int steps;
            if (tsteps > psteps)
            {
                steps = tsteps < 512 ? tsteps : 512;
            }
            else
            {
                steps = psteps < 128 ? psteps : 128;
            }
            steps = steps > 1 ? steps : 1;
            dphi /= steps;
            dtheta /= steps;
            for(int a=2; a<steps-2; a++)
            {
                DrawPoint(theta1 + a * dtheta, phi1 + a * dphi, brightness);
            }
        }

        public void ClearBuffer()
        {
            dotsused = 0;
            usingScreenBuffer = false;
        }

        private void prepareScreenBuffer()
        {
            for (int a = 0; a < screenBuffer.Length; a++)
            {
                screenBuffer[a] = 0;
            }
            usingScreenBuffer = true;
            for (int a = 0; a < dotsused; a++)
            {
                int bitOffset = a * 18;
                int byteOffset = bitOffset / 8;
                uint res = dotBuffer[byteOffset];
                res <<= 8;
                res |= dotBuffer[byteOffset + 1];
                res <<= 8;
                res |= dotBuffer[byteOffset + 2];
                res <<= (bitOffset % 8);
                addPoint((res & 0xFF1000) >> 15, (res & 0x007F00) >> 8, (byte)((res & 0x0000C0) >> 6));
            }
        }

        private void addPoint(uint theta, uint phi, byte brightness)
        {
            //this function takes theta in pixels from south (0) towards east (+), phi in pixels from zenith (0) down to horizon (+)
            theta %= 512;
            phi %= 128;
            brightness %= 4;
            if (!usingScreenBuffer)
            {
                //screen buffer must be used at 16384 bytes, and dots must be used below 16384 bytes
                //7281 dots = 16382.25 bytes
                if (dotsused >= 7281)
                {
                    prepareScreenBuffer();
                }
                else
                {
                    int bitOffset = dotsused * 18;
                    int byteOffset = bitOffset / 8;
                    uint wr = (theta << 15) | (phi << 8) | ((uint)brightness << 6);
                    wr >>= (bitOffset % 8);
                    uint res = dotBuffer[byteOffset];
                    res &= ((uint)0xFF << (8-(bitOffset % 8)));
                    res |= (wr & 0xFF0000) >> 16;
                    dotBuffer[byteOffset] = (byte)(res&0xFF);
                    dotBuffer[byteOffset + 1] = (byte)((wr >> 8) & 0xFF);
                    dotBuffer[byteOffset + 2] = (byte)((wr) & 0xFF);
                    dotsused++;
                }
            }
            if (usingScreenBuffer)
            {
                int bitOffset = (int)(phi * 1024 + theta * 2);
                int byteOffset = bitOffset / 8;
                byte res = screenBuffer[byteOffset];
                int bitshift = (6 - (bitOffset % 8));
                int cur = (res >> bitshift) & 0x03;
                if (brightness > cur)
                {
                    if (cur == 1)
                        res &= (byte)(~(1 << bitshift));
                    res |= (byte)(brightness << bitshift);
                    screenBuffer[byteOffset] = res;
                }
                //dotsused++;
            }
        }


        public async Task UpdateFlash(string fn, uint len)
        {
            if (_isConnected)
                Disconnect();

            try
            {
                device = new Socket(SocketType.Stream, ProtocolType.Tcp);
                //device.SendBufferSize = 16400;
                //device.SendTimeout = 2000;
                //device.ReceiveTimeout = 2000;
                device.Connect(lpip, 80);
                device.Send(Encoding.ASCII.GetBytes(flash_request));
                if (receiveUint() == connection_response_code)
                {
                    _isConnected = true;
                    sendUint((uint)len);
                    await Task.Run(() => { device.SendFile(fn); });
                    if (receiveUint() == len) //success
                    {
                        if (MessageBox.Show("File successfully transferred. Click OK to begin writing to memory.", "Firmware Transfer Success", MessageBoxButton.OKCancel) == MessageBoxResult.OK)
                            sendUint(flash_write_code);
                        else
                            sendUint(0);
                    }
                    else
                    {
                        int av = device.Available;
                        byte[] inbuf = new byte[1024];
                        int rd = av > 1024 ? 1024 : av;
                        device.Receive(inbuf, rd, SocketFlags.None);
                        string msg = Encoding.ASCII.GetString(inbuf, 0, rd);
                        MessageBox.Show(msg, "Firmware Transfer Error");
                    }
                }
                else
                {
                    _isConnected = false;
                    device.Disconnect(false);
                }
                //device = new SerialPort("COM3", 115200);
                //device.Open();
                //device.Write("S");//start operation
            }
            catch (Exception x)
            {
                _isConnected = false;
                device = null;
                MessageBox.Show(x.ToString(), "Firmware Transfer Exception");
            }
        }
    }
}
